--[[
    ╔══════════════════════════════════════════════════╗
    ║      CFX-NOTIFICAÇÕES - Script Cliente           ║
    ║      Sistema de Notificações para FiveM          ║
    ║                  by CfxAI                        ║
    ╚══════════════════════════════════════════════════╝
]]

-- Obtém o objeto principal do QBCore
local QBCore = exports['qb-core']:GetCoreObject()

-- ════════════════════════════════════════════════════
-- Função principal para enviar notificação ao NUI
-- ════════════════════════════════════════════════════

--- Envia uma notificação para a interface do jogador
--- @param type string - Tipo da notificação: 'success', 'error', 'info', 'warning'
--- @param title string - Título da notificação
--- @param message string - Mensagem da notificação
--- @param duration number|nil - Duração em milissegundos (opcional, usa Config.DefaultDuration)
local function SendNotification(type, title, message, duration)
    -- Validação do tipo de notificação
    local notifType = type or 'info'
    if not Config.Types[notifType] then
        notifType = 'info'
    end

    -- Usa o título padrão do tipo se nenhum for fornecido
    local notifTitle = title
    if not notifTitle or notifTitle == '' then
        notifTitle = Config.Types[notifType].label
    end

    -- Validação da mensagem
    local notifMessage = message or ''

    -- Duração padrão se não especificada
    local notifDuration = duration or Config.DefaultDuration

    -- Envia os dados para o NUI (interface HTML)
    SendNUIMessage({
        action = 'addNotification',
        type = notifType,
        title = notifTitle,
        message = notifMessage,
        duration = notifDuration,
        icon = Config.Types[notifType].icon,
        color = Config.Types[notifType].color,
        position = Config.Position,
        maxNotifications = Config.MaxNotifications,
        playSound = Config.PlaySound,
    })
end

-- ════════════════════════════════════════════════════
-- Eventos do cliente
-- ════════════════════════════════════════════════════

--- Evento para receber notificações disparadas pelo servidor
RegisterNetEvent('cfx-notificacoes:client:notify', function(type, title, message, duration)
    SendNotification(type, title, message, duration)
end)

-- ════════════════════════════════════════════════════
-- Exportações para uso por outros recursos
-- ════════════════════════════════════════════════════

--- Export: Permite que outros resources enviem notificações
--- Exemplo de uso: exports['cfx-notificacoes']:Notify('success', 'Título', 'Mensagem', 5000)
exports('Notify', function(type, title, message, duration)
    SendNotification(type, title, message, duration)
end)

-- ════════════════════════════════════════════════════
-- Comando de teste
-- ════════════════════════════════════════════════════

--- Comando para testar as notificações no jogo
--- Uso: /notificar [tipo] [mensagem]
--- Exemplo: /notificar success Operação realizada com sucesso!
if Config.TestCommand then
    RegisterCommand(Config.TestCommand, function(source, args)
        -- Verifica se o tipo foi fornecido
        local type = args[1] or 'info'

        -- Monta a mensagem a partir dos argumentos restantes
        local messageParts = {}
        for i = 2, #args do
            messageParts[#messageParts + 1] = args[i]
        end
        local message = table.concat(messageParts, ' ')

        -- Se nenhuma mensagem foi informada, usa uma mensagem padrão
        if message == '' then
            message = 'Esta é uma notificação de teste do tipo: ' .. type
        end

        -- Obtém o título padrão do tipo de notificação
        local title = nil
        if Config.Types[type] then
            title = Config.Types[type].label
        end

        -- Envia a notificação
        SendNotification(type, title, message)
    end, false)
end

-- ════════════════════════════════════════════════════
-- Integração com o sistema de notificações do QBCore
-- ════════════════════════════════════════════════════

--- Substitui a notificação padrão do QBCore (opcional)
--- Descomente as linhas abaixo para substituir o sistema padrão
--[[
RegisterNetEvent('QBCore:Notify', function(text, type, duration)
    local notifType = type or 'info'

    -- Mapeia os tipos do QBCore para os tipos deste sistema
    local typeMap = {
        ['primary'] = 'info',
        ['inform'] = 'info',
        ['success'] = 'success',
        ['error'] = 'error',
        ['warning'] = 'warning',
    }

    notifType = typeMap[notifType] or 'info'
    SendNotification(notifType, nil, text, duration)
end)
]]

-- Mensagem de confirmação no console ao iniciar o resource
print('^2[cfx-notificacoes]^0 Script cliente carregado com sucesso!')
