/*
    ╔══════════════════════════════════════════════════╗
    ║      CFX-NOTIFICAÇÕES - Script JavaScript        ║
    ║      Sistema de Notificações para FiveM          ║
    ║                  by CfxAI                        ║
    ╚══════════════════════════════════════════════════╝
*/

(function () {
    'use strict';

    // ═══════════════════════════════════════════════════
    // Variáveis e configurações
    // ═══════════════════════════════════════════════════

    const container = document.getElementById('notification-container');
    let notificationQueue = [];
    let maxNotifications = 5;
    let currentPosition = 'top-right';

    // Contador único para IDs das notificações
    let notificationIdCounter = 0;

    // ═══════════════════════════════════════════════════
    // Determinar animações baseadas na posição
    // ═══════════════════════════════════════════════════

    /**
     * Retorna a classe de animação de entrada baseada na posição
     * @param {string} position - Posição do container
     * @returns {string} Classe CSS de animação de entrada
     */
    function getSlideInClass(position) {
        if (position.includes('right')) return 'slide-in-right';
        if (position.includes('left')) return 'slide-in-left';
        return 'slide-in-center';
    }

    /**
     * Retorna a classe de animação de saída baseada na posição
     * @param {string} position - Posição do container
     * @returns {string} Classe CSS de animação de saída
     */
    function getSlideOutClass(position) {
        if (position.includes('right')) return 'slide-out-right';
        if (position.includes('left')) return 'slide-out-left';
        return 'slide-out-center';
    }

    // ═══════════════════════════════════════════════════
    // Atualizar posição do container
    // ═══════════════════════════════════════════════════

    /**
     * Atualiza a posição do container de notificações
     * @param {string} position - Nova posição
     */
    function updatePosition(position) {
        // Remove todas as classes de posição anteriores
        container.classList.remove(
            'top-right', 'top-left',
            'bottom-right', 'bottom-left',
            'top-center', 'bottom-center'
        );

        // Adiciona a nova classe de posição
        currentPosition = position || 'top-right';
        container.classList.add(currentPosition);
    }

    // ═══════════════════════════════════════════════════
    // Remover notificação
    // ═══════════════════════════════════════════════════

    /**
     * Remove uma notificação com animação de saída
     * @param {HTMLElement} element - Elemento da notificação
     * @param {number} id - ID da notificação
     */
    function removeNotification(element, id) {
        // Evita remoção duplicada
        if (element.dataset.removing === 'true') return;
        element.dataset.removing = 'true';

        // Adiciona a animação de saída
        const slideInClass = getSlideInClass(currentPosition);
        const slideOutClass = getSlideOutClass(currentPosition);

        element.classList.remove(slideInClass);
        element.classList.add(slideOutClass);

        // Remove o elemento do DOM após a animação
        setTimeout(function () {
            if (element.parentNode) {
                element.parentNode.removeChild(element);
            }

            // Remove da fila
            notificationQueue = notificationQueue.filter(function (n) {
                return n.id !== id;
            });
        }, 350);
    }

    // ═══════════════════════════════════════════════════
    // Criar notificação
    // ═══════════════════════════════════════════════════

    /**
     * Cria e exibe uma nova notificação
     * @param {Object} data - Dados da notificação
     * @param {string} data.type - Tipo: success, error, info, warning
     * @param {string} data.title - Título da notificação
     * @param {string} data.message - Mensagem da notificação
     * @param {number} data.duration - Duração em milissegundos
     * @param {string} data.icon - Classe do ícone Font Awesome
     * @param {string} data.position - Posição do container
     * @param {number} data.maxNotifications - Máximo de notificações visíveis
     * @param {boolean} data.playSound - Se deve tocar som
     */
    function createNotification(data) {
        const type = data.type || 'info';
        const title = data.title || 'Notificação';
        const message = data.message || '';
        const duration = data.duration || 5000;
        const icon = data.icon || 'fa-info-circle';
        const position = data.position || 'top-right';

        // Atualiza configurações
        if (data.maxNotifications) {
            maxNotifications = data.maxNotifications;
        }

        // Atualiza posição do container
        updatePosition(position);

        // Verifica o limite de notificações e remove as mais antigas
        while (notificationQueue.length >= maxNotifications) {
            const oldest = notificationQueue[0];
            if (oldest && oldest.element) {
                removeNotification(oldest.element, oldest.id);
            }
            notificationQueue.shift();
        }

        // Gera um ID único para esta notificação
        const notifId = ++notificationIdCounter;

        // Cria o elemento da notificação
        const notifEl = document.createElement('div');
        notifEl.className = 'notification ' + type + ' ' + getSlideInClass(position);
        notifEl.dataset.id = notifId;

        // Monta o HTML interno da notificação
        notifEl.innerHTML =
            '<div class="notification-icon">' +
                '<i class="fas ' + escapeHtml(icon) + '"></i>' +
            '</div>' +
            '<div class="notification-content">' +
                '<div class="notification-title">' + escapeHtml(title) + '</div>' +
                (message ? '<div class="notification-message">' + escapeHtml(message) + '</div>' : '') +
            '</div>' +
            '<span class="cfx-accent">CfxAI</span>' +
            '<button class="notification-close"><i class="fas fa-times"></i></button>' +
            '<div class="notification-progress">' +
                '<div class="notification-progress-bar" style="animation-duration: ' + duration + 'ms;"></div>' +
            '</div>';

        // Evento de clique para fechar a notificação
        notifEl.addEventListener('click', function () {
            clearTimeout(autoRemoveTimer);
            removeNotification(notifEl, notifId);
        });

        // Botão de fechar (previne propagação para não duplicar)
        var closeBtn = notifEl.querySelector('.notification-close');
        if (closeBtn) {
            closeBtn.addEventListener('click', function (e) {
                e.stopPropagation();
                clearTimeout(autoRemoveTimer);
                removeNotification(notifEl, notifId);
            });
        }

        // Adiciona ao container
        container.appendChild(notifEl);

        // Adiciona à fila de controle
        notificationQueue.push({
            id: notifId,
            element: notifEl
        });

        // Timer para remoção automática
        var autoRemoveTimer = setTimeout(function () {
            removeNotification(notifEl, notifId);
        }, duration);

        // Pausa o timer ao passar o mouse sobre a notificação
        notifEl.addEventListener('mouseenter', function () {
            clearTimeout(autoRemoveTimer);
            var progressBar = notifEl.querySelector('.notification-progress-bar');
            if (progressBar) {
                progressBar.style.animationPlayState = 'paused';
            }
        });

        // Retoma o timer ao remover o mouse
        notifEl.addEventListener('mouseleave', function () {
            var progressBar = notifEl.querySelector('.notification-progress-bar');
            if (progressBar) {
                progressBar.style.animationPlayState = 'running';
            }
            // Reinicia o timer com o tempo restante estimado (usa duração total como fallback)
            autoRemoveTimer = setTimeout(function () {
                removeNotification(notifEl, notifId);
            }, duration * 0.5);
        });
    }

    // ═══════════════════════════════════════════════════
    // Utilitários
    // ═══════════════════════════════════════════════════

    /**
     * Escapa caracteres HTML para prevenir XSS
     * @param {string} text - Texto para escapar
     * @returns {string} Texto escapado
     */
    function escapeHtml(text) {
        if (!text) return '';
        var div = document.createElement('div');
        div.appendChild(document.createTextNode(text));
        return div.innerHTML;
    }

    // ═══════════════════════════════════════════════════
    // Listener de mensagens do NUI (FiveM)
    // ═══════════════════════════════════════════════════

    window.addEventListener('message', function (event) {
        var data = event.data;

        if (!data || !data.action) return;

        switch (data.action) {
            // Adiciona uma nova notificação
            case 'addNotification':
                createNotification(data);
                break;

            // Mostra o container de notificações
            case 'show':
                container.style.display = 'flex';
                break;

            // Esconde o container de notificações
            case 'hide':
                container.style.display = 'none';
                break;

            // Remove todas as notificações
            case 'clearAll':
                notificationQueue.forEach(function (notif) {
                    if (notif.element) {
                        removeNotification(notif.element, notif.id);
                    }
                });
                notificationQueue = [];
                break;
        }
    });

    // ═══════════════════════════════════════════════════
    // Inicialização
    // ═══════════════════════════════════════════════════

    // Garante que o container está visível na inicialização
    container.style.display = 'flex';

})();
