--[[
    ╔══════════════════════════════════════════════════╗
    ║      CFX-NOTIFICAÇÕES - Script Servidor          ║
    ║      Sistema de Notificações para FiveM          ║
    ║                  by CfxAI                        ║
    ╚══════════════════════════════════════════════════╝
]]

-- Obtém o objeto principal do QBCore
local QBCore = exports['qb-core']:GetCoreObject()

-- ════════════════════════════════════════════════════
-- Funções auxiliares
-- ════════════════════════════════════════════════════

--- Verifica se o jogador possui permissão de administrador
--- @param source number - ID do jogador no servidor
--- @return boolean - Retorna true se o jogador for admin
local function IsPlayerAdmin(source)
    local Player = QBCore.Functions.GetPlayer(source)
    if not Player then return false end

    -- Verifica o grupo de permissão do jogador
    if QBCore.Functions.HasPermission(source, Config.AdminGroup) then
        return true
    end

    -- Verifica se o jogador pertence ao grupo 'god' (superadmin)
    if QBCore.Functions.HasPermission(source, 'god') then
        return true
    end

    return false
end

-- ════════════════════════════════════════════════════
-- Funções de notificação do servidor
-- ════════════════════════════════════════════════════

--- Envia uma notificação para um jogador específico
--- @param targetSource number - ID do jogador alvo
--- @param type string - Tipo da notificação: 'success', 'error', 'info', 'warning'
--- @param title string - Título da notificação
--- @param message string - Mensagem da notificação
--- @param duration number|nil - Duração em milissegundos (opcional)
local function NotifyPlayer(targetSource, type, title, message, duration)
    TriggerClientEvent('cfx-notificacoes:client:notify', targetSource, type, title, message, duration)
end

--- Envia uma notificação para todos os jogadores do servidor
--- @param type string - Tipo da notificação: 'success', 'error', 'info', 'warning'
--- @param title string - Título da notificação
--- @param message string - Mensagem da notificação
--- @param duration number|nil - Duração em milissegundos (opcional)
local function NotifyAll(type, title, message, duration)
    TriggerClientEvent('cfx-notificacoes:client:notify', -1, type, title, message, duration)
end

-- ════════════════════════════════════════════════════
-- Exportações para uso por outros recursos
-- ════════════════════════════════════════════════════

--- Export: Envia notificação para um jogador específico
--- Exemplo: exports['cfx-notificacoes']:NotifyPlayer(source, 'success', 'Título', 'Mensagem', 5000)
exports('NotifyPlayer', function(targetSource, type, title, message, duration)
    NotifyPlayer(targetSource, type, title, message, duration)
end)

--- Export: Envia notificação para todos os jogadores
--- Exemplo: exports['cfx-notificacoes']:NotifyAll('info', 'Anúncio', 'Mensagem para todos')
exports('NotifyAll', function(type, title, message, duration)
    NotifyAll(type, title, message, duration)
end)

-- ════════════════════════════════════════════════════
-- Comandos do servidor
-- ════════════════════════════════════════════════════

--- Comando de anúncio para administradores
--- Envia uma notificação para todos os jogadores conectados
--- Uso: /anuncio [mensagem]
--- Exemplo: /anuncio Manutenção do servidor em 10 minutos!
if Config.AnnounceCommand then
    RegisterCommand(Config.AnnounceCommand, function(source, args)
        -- Verifica se o comando foi executado pelo console do servidor (source 0)
        if source == 0 then
            -- Monta a mensagem a partir dos argumentos
            local message = table.concat(args, ' ')
            if message == '' then
                print('[cfx-notificacoes] Uso correto: /' .. Config.AnnounceCommand .. ' [mensagem]')
                return
            end

            NotifyAll('warning', 'Anúncio do Servidor', message, 8000)
            print('[cfx-notificacoes] Anúncio enviado para todos: ' .. message)
            return
        end

        -- Verifica se o jogador tem permissão de administrador
        if not IsPlayerAdmin(source) then
            NotifyPlayer(source, 'error', 'Sem Permissão', 'Você não tem permissão para usar este comando.', 5000)
            return
        end

        -- Monta a mensagem a partir dos argumentos
        local message = table.concat(args, ' ')
        if message == '' then
            NotifyPlayer(source, 'error', 'Erro', 'Uso correto: /' .. Config.AnnounceCommand .. ' [mensagem]', 5000)
            return
        end

        -- Obtém o nome do administrador
        local Player = QBCore.Functions.GetPlayer(source)
        local adminName = 'Administrador'
        if Player then
            adminName = Player.PlayerData.charinfo.firstname .. ' ' .. Player.PlayerData.charinfo.lastname
        end

        -- Envia a notificação para todos os jogadores
        NotifyAll('warning', 'Anúncio do Servidor', message, 8000)

        -- Confirma o envio para o administrador
        NotifyPlayer(source, 'success', 'Anúncio Enviado', 'Sua mensagem foi enviada para todos os jogadores.', 3000)

        -- Log no console do servidor
        print('[cfx-notificacoes] Anúncio de ' .. adminName .. ' (ID: ' .. source .. '): ' .. message)
    end, false)
end

-- ════════════════════════════════════════════════════
-- Eventos adicionais do servidor
-- ════════════════════════════════════════════════════

--- Evento para enviar notificação de boas-vindas quando um jogador entrar
--- Descomente para ativar a mensagem de boas-vindas
--[[
RegisterNetEvent('QBCore:Server:PlayerLoaded', function(Player)
    local src = Player.PlayerData.source
    local firstName = Player.PlayerData.charinfo.firstname

    -- Notificação de boas-vindas para o jogador
    Wait(3000) -- Aguarda 3 segundos para garantir que o NUI carregou
    NotifyPlayer(src, 'success', 'Bem-vindo!', 'Olá ' .. firstName .. ', bem-vindo ao servidor!', 7000)

    -- Notificação para todos os outros jogadores
    NotifyAll('info', 'Novo Jogador', firstName .. ' entrou no servidor.', 5000)
end)
]]

-- Mensagem de confirmação no console ao iniciar o resource
print('^2[cfx-notificacoes]^0 Script servidor carregado com sucesso!')
