--[[
    cfx-npc-shop | Client
    Sistema de Loja com NPC para FiveM (ESX Legacy)
    Desenvolvido por CfxAI
]]

-- ============================================================================
-- VARIAVEIS GLOBAIS
-- ============================================================================

local ESX = exports['es_extended']:getSharedObject()

local spawnedPeds = {}   -- Tabela com referencias aos NPCs criados
local shopBlips   = {}   -- Tabela com referencias aos blips criados
local isMenuOpen  = false -- Controla se o menu esta aberto (evita abrir varias vezes)

-- ============================================================================
-- FUNCOES AUXILIARES
-- ============================================================================

--- Desenha texto 3D flutuante no mundo
--- @param coords vector3  Posicao no mundo
--- @param text   string   Texto a ser exibido
local function DrawText3D(coords, text)
    local onScreen, screenX, screenY = World3dToScreen2d(coords.x, coords.y, coords.z + 1.0)

    if onScreen then
        SetTextScale(0.35, 0.35)
        SetTextFont(4)
        SetTextProportional(true)
        SetTextColour(255, 255, 255, 215)
        SetTextDropshadow(0, 0, 0, 0, 255)
        SetTextEdge(2, 0, 0, 0, 150)
        SetTextDropShadow()
        SetTextOutline()
        SetTextEntry('STRING')
        SetTextCentre(true)
        AddTextComponentString(text)
        DrawText(screenX, screenY)
    end
end

--- Carrega um modelo de ped e aguarda ate estar pronto
--- @param model string|number  Hash ou nome do modelo
local function LoadModel(model)
    local hash = type(model) == 'string' and joaat(model) or model

    RequestModel(hash)

    local timeout = 0
    while not HasModelLoaded(hash) do
        Wait(10)
        timeout = timeout + 10
        if timeout > 5000 then
            print(('[cfx-npc-shop] ERRO: Timeout ao carregar modelo: %s'):format(model))
            return false
        end
    end

    return true
end

--- Formata um valor monetario com o simbolo da config
--- @param value number  Valor a ser formatado
--- @return string
local function FormatPrice(value)
    return ('%s %s'):format(Config.CurrencySymbol, tostring(value))
end

-- ============================================================================
-- CRIACAO DE NPCS E BLIPS
-- ============================================================================

--- Cria todos os NPCs e blips definidos na configuracao
local function SetupShops()
    for shopIndex, shop in ipairs(Config.Shops) do
        -- ----- BLIP -----
        local blipData = shop.blip
        local pedData  = shop.ped
        local blip     = AddBlipForCoord(pedData.coords.x, pedData.coords.y, pedData.coords.z)

        SetBlipSprite(blip, blipData.sprite)
        SetBlipDisplay(blip, 4)
        SetBlipScale(blip, blipData.scale)
        SetBlipColour(blip, blipData.color)
        SetBlipAsShortRange(blip, true)
        BeginTextCommandSetBlipName('STRING')
        AddTextComponentString(blipData.label)
        EndTextCommandSetBlipName(blip)

        shopBlips[#shopBlips + 1] = blip

        -- ----- NPC -----
        if LoadModel(pedData.model) then
            local hash = joaat(pedData.model)
            local ped  = CreatePed(4, hash, pedData.coords.x, pedData.coords.y, pedData.coords.z - 1.0, pedData.coords.w, false, true)

            SetEntityHeading(ped, pedData.coords.w)
            FreezeEntityPosition(ped, true)
            SetEntityInvincible(ped, true)
            SetBlockingOfNonTemporaryEvents(ped, true)

            -- Aplica cenario (animacao idle)
            if pedData.scenario then
                TaskStartScenarioInPlace(ped, pedData.scenario, 0, true)
            end

            spawnedPeds[#spawnedPeds + 1] = {
                entity    = ped,
                shopIndex = shopIndex
            }

            SetModelAsNoLongerNeeded(hash)
        end

        -- ----- OX_TARGET (opcional) -----
        if Config.UseTarget then
            exports.ox_target:addLocalEntity(spawnedPeds[#spawnedPeds].entity, {
                {
                    name    = ('cfx_shop_%d'):format(shopIndex),
                    icon    = 'fas fa-shopping-basket',
                    label   = ('Abrir %s'):format(shop.name),
                    onSelect = function()
                        OpenShopMenu(shopIndex)
                    end
                }
            })
        end
    end
end

-- ============================================================================
-- MENU DA LOJA
-- ============================================================================

--- Abre o menu de uma loja especifica
--- @param shopIndex number  Indice da loja em Config.Shops
function OpenShopMenu(shopIndex)
    if isMenuOpen then return end

    local shop = Config.Shops[shopIndex]
    if not shop then return end

    isMenuOpen = true

    -- Construir os elementos do menu
    local elements = {}

    for _, item in ipairs(shop.items) do
        elements[#elements + 1] = {
            unselectable = false,
            icon         = 'fas fa-box',
            title        = item.label,
            description  = ('Preco: %s'):format(FormatPrice(item.price)),
            value        = item.name
        }
    end

    -- Menu usando esx_menu_default (padrao do ESX Legacy)
    ESX.OpenContext('right', elements, function(menu, element)
        -- Submenu para confirmar a compra e escolher forma de pagamento
        local confirmElements = {
            {
                unselectable = true,
                icon  = 'fas fa-info-circle',
                title = ('Comprar: %s'):format(element.title),
                description = ('Preco: %s'):format(FormatPrice(GetItemPrice(shopIndex, element.value)))
            },
            {
                icon  = 'fas fa-money-bill-wave',
                title = 'Pagar com Dinheiro',
                description = 'Usar dinheiro em especie',
                value = 'cash'
            },
            {
                icon  = 'fas fa-credit-card',
                title = 'Pagar com Banco',
                description = 'Usar conta bancaria',
                value = 'bank'
            }
        }

        ESX.OpenContext('right', confirmElements, function(menu2, element2)
            if element2.value == 'cash' or element2.value == 'bank' then
                -- Enviar evento de compra para o servidor
                TriggerServerEvent('cfx-npc-shop:buy', shopIndex, element.value, element2.value)
                ESX.CloseContext()
                isMenuOpen = false
            end
        end, function()
            -- Voltar ao menu da loja ao cancelar o submenu
            isMenuOpen = false
            OpenShopMenu(shopIndex)
        end)
    end, function()
        -- Fechar menu completamente
        ESX.CloseContext()
        isMenuOpen = false
    end)
end

--- Retorna o preco de um item de uma loja especifica
--- @param shopIndex number  Indice da loja
--- @param itemName  string  Nome interno do item
--- @return number
function GetItemPrice(shopIndex, itemName)
    local shop = Config.Shops[shopIndex]
    if not shop then return 0 end

    for _, item in ipairs(shop.items) do
        if item.name == itemName then
            return item.price
        end
    end

    return 0
end

-- ============================================================================
-- LOOP PRINCIPAL - INTERACAO COM TECLA
-- ============================================================================

-- Thread que verifica a proximidade do jogador com os NPCs
-- Ativa apenas quando Config.UseTarget = false
CreateThread(function()
    -- Aguardar carregamento completo do jogador
    while not ESX.IsPlayerLoaded() do
        Wait(500)
    end

    -- Criar lojas apos o jogador carregar
    SetupShops()

    -- Se estiver usando ox_target, nao e necessario o loop de interacao
    if Config.UseTarget then return end

    while true do
        local sleepTime  = 500  -- Tempo de espera padrao (longe de qualquer NPC)
        local playerPed  = PlayerPedId()
        local playerPos  = GetEntityCoords(playerPed)
        local nearShop   = nil

        -- Verificar distancia para cada NPC
        for _, data in ipairs(spawnedPeds) do
            local pedCoords = GetEntityCoords(data.entity)
            local distance  = #(playerPos - pedCoords)

            if distance < 10.0 then
                sleepTime = 0 -- Atualizar a cada frame quando perto

                if distance < Config.InteractionDistance then
                    nearShop = data.shopIndex

                    -- Desenhar texto 3D acima do NPC
                    local shopName = Config.Shops[data.shopIndex].name
                    DrawText3D(pedCoords, ('~g~[E]~s~ %s'):format(shopName))

                    -- Verificar se a tecla de interacao foi pressionada
                    if IsControlJustPressed(0, Config.InteractionKey) and not isMenuOpen then
                        OpenShopMenu(data.shopIndex)
                    end

                    break -- Interagir apenas com o NPC mais proximo
                end
            end
        end

        Wait(sleepTime)
    end
end)

-- ============================================================================
-- LIMPEZA AO PARAR O RESOURCE
-- ============================================================================

AddEventHandler('onResourceStop', function(resourceName)
    if resourceName ~= GetCurrentResourceName() then return end

    -- Remover todos os NPCs criados
    for _, data in ipairs(spawnedPeds) do
        if DoesEntityExist(data.entity) then
            DeleteEntity(data.entity)
        end
    end

    -- Remover todos os blips criados
    for _, blip in ipairs(shopBlips) do
        if DoesBlipExist(blip) then
            RemoveBlip(blip)
        end
    end

    -- Fechar menu se estiver aberto
    if isMenuOpen then
        ESX.CloseContext()
        isMenuOpen = false
    end
end)

-- ============================================================================
-- NOTIFICACOES DO SERVIDOR
-- ============================================================================

--- Recebe notificacao de compra bem-sucedida
RegisterNetEvent('cfx-npc-shop:purchaseSuccess', function(itemLabel, price)
    ESX.ShowNotification(('~g~Compra realizada! ~s~Voce comprou ~b~%s~s~ por %s'):format(itemLabel, FormatPrice(price)))
end)

--- Recebe notificacao de erro na compra
RegisterNetEvent('cfx-npc-shop:purchaseError', function(message)
    ESX.ShowNotification(('~r~Erro: ~s~%s'):format(message))
end)
