--[[
    cfx-npc-shop | Server
    Sistema de Loja com NPC para FiveM (ESX Legacy)
    Desenvolvido por CfxAI
]]

-- ============================================================================
-- INICIALIZACAO DO ESX
-- ============================================================================

local ESX = exports['es_extended']:getSharedObject()

-- ============================================================================
-- FUNCOES AUXILIARES
-- ============================================================================

--- Busca um item na configuracao de uma loja e retorna seus dados
--- Isso serve como validacao anti-exploit, garantindo que o item existe
--- e que o preco corresponde ao que esta na config do servidor
--- @param shopIndex number  Indice da loja em Config.Shops
--- @param itemName  string  Nome interno do item
--- @return table|nil        Dados do item ou nil se nao encontrado
local function GetShopItem(shopIndex, itemName)
    -- Validar se o indice da loja e valido
    if not shopIndex or type(shopIndex) ~= 'number' then return nil end
    if shopIndex < 1 or shopIndex > #Config.Shops then return nil end

    local shop = Config.Shops[shopIndex]
    if not shop then return nil end

    -- Procurar o item na lista de itens da loja
    for _, item in ipairs(shop.items) do
        if item.name == itemName then
            return item
        end
    end

    return nil
end

--- Registra uma tentativa de exploit no console do servidor
--- @param source number  ID do jogador
--- @param reason string  Motivo da suspeita
local function LogExploit(source, reason)
    local xPlayer = ESX.GetPlayerFromId(source)
    local name    = xPlayer and xPlayer.getName() or 'Desconhecido'
    local ids     = xPlayer and xPlayer.getIdentifier() or 'N/A'

    print(('[cfx-npc-shop] ^1ANTI-EXPLOIT^0 | Jogador: %s (ID: %d, Identifier: %s) | Motivo: %s'):format(
        name, source, ids, reason
    ))
end

-- ============================================================================
-- EVENTO DE COMPRA
-- ============================================================================

--- Evento principal de compra disparado pelo cliente
--- Validacoes realizadas:
---   1. O indice da loja existe na configuracao
---   2. O item existe na loja especificada
---   3. O metodo de pagamento e valido (cash ou bank)
---   4. O jogador possui saldo suficiente
---   5. O item e adicionado ao inventario do jogador
RegisterNetEvent('cfx-npc-shop:buy', function(shopIndex, itemName, paymentMethod)
    local source  = source
    local xPlayer = ESX.GetPlayerFromId(source)

    -- Verificar se o jogador e valido
    if not xPlayer then
        print(('[cfx-npc-shop] ERRO: Jogador com ID %d nao encontrado no ESX'):format(source))
        return
    end

    -- ---- VALIDACAO 1: Indice da loja ----
    if not shopIndex or type(shopIndex) ~= 'number' or shopIndex < 1 or shopIndex > #Config.Shops then
        LogExploit(source, ('Indice de loja invalido: %s'):format(tostring(shopIndex)))
        TriggerClientEvent('cfx-npc-shop:purchaseError', source, 'Loja invalida.')
        return
    end

    -- ---- VALIDACAO 2: Item existe na loja ----
    local item = GetShopItem(shopIndex, itemName)

    if not item then
        LogExploit(source, ('Item inexistente: "%s" na loja %d'):format(tostring(itemName), shopIndex))
        TriggerClientEvent('cfx-npc-shop:purchaseError', source, 'Item nao encontrado nesta loja.')
        return
    end

    -- ---- VALIDACAO 3: Metodo de pagamento ----
    if paymentMethod ~= 'cash' and paymentMethod ~= 'bank' then
        LogExploit(source, ('Metodo de pagamento invalido: "%s"'):format(tostring(paymentMethod)))
        TriggerClientEvent('cfx-npc-shop:purchaseError', source, 'Metodo de pagamento invalido.')
        return
    end

    -- ---- VALIDACAO 4: Saldo suficiente ----
    local price        = item.price -- Preco obtido da config do SERVIDOR (anti-exploit)
    local playerMoney  = 0

    if paymentMethod == 'cash' then
        playerMoney = xPlayer.getMoney()
    else
        playerMoney = xPlayer.getAccount('bank').money
    end

    if playerMoney < price then
        local methodLabel = paymentMethod == 'cash' and 'dinheiro' or 'banco'
        TriggerClientEvent('cfx-npc-shop:purchaseError', source,
            ('Saldo insuficiente! Voce precisa de %s %d (%s).'):format(
                Config.CurrencySymbol, price, methodLabel
            )
        )
        return
    end

    -- ---- PROCESSAR COMPRA ----

    -- Remover dinheiro do jogador
    if paymentMethod == 'cash' then
        xPlayer.removeMoney(price, ('Compra: %s'):format(item.label))
    else
        xPlayer.removeAccountMoney('bank', price, ('Compra: %s'):format(item.label))
    end

    -- Adicionar item ao inventario
    xPlayer.addInventoryItem(item.name, 1)

    -- Notificar o jogador sobre a compra bem-sucedida
    TriggerClientEvent('cfx-npc-shop:purchaseSuccess', source, item.label, price)

    -- Log no console do servidor
    print(('[cfx-npc-shop] %s (ID: %d) comprou %s por %s %d (%s)'):format(
        xPlayer.getName(),
        source,
        item.label,
        Config.CurrencySymbol,
        price,
        paymentMethod == 'cash' and 'dinheiro' or 'banco'
    ))
end)

-- ============================================================================
-- MENSAGEM DE INICIALIZACAO
-- ============================================================================

AddEventHandler('onResourceStart', function(resourceName)
    if resourceName ~= GetCurrentResourceName() then return end

    local totalItems = 0
    for _, shop in ipairs(Config.Shops) do
        totalItems = totalItems + #shop.items
    end

    print(('[cfx-npc-shop] ^2Carregado com sucesso!^0 %d loja(s) | %d item(ns) registrados'):format(
        #Config.Shops, totalItems
    ))
end)
