// ============================================================
// cfx-velocimetro | Script NUI
// Velocimetro HUD moderno para FiveM
// by CfxAI
// ============================================================

(function () {
    'use strict';

    // Referencias aos elementos do DOM
    const velocimetro = document.getElementById('velocimetro');
    const velocidadeEl = document.getElementById('velocidade');
    const marchaEl = document.getElementById('marcha');
    const rpmBarra = document.getElementById('rpm-barra');
    const combustivelBarra = document.getElementById('combustivel-barra');
    const combustivelValor = document.getElementById('combustivel-valor');
    const indMotor = document.getElementById('ind-motor');
    const indLuzes = document.getElementById('ind-luzes');

    // Velocidade atual exibida (para transicao suave)
    let velocidadeAtual = 0;
    let velocidadeAlvo = 0;
    let animacaoId = null;

    // ============================================================
    // Funcao para interpolar a velocidade suavemente
    // ============================================================
    function animarVelocidade() {
        if (Math.abs(velocidadeAtual - velocidadeAlvo) < 1) {
            velocidadeAtual = velocidadeAlvo;
            velocidadeEl.textContent = Math.round(velocidadeAtual);
            animacaoId = null;
            return;
        }

        // Interpolacao linear com fator de suavizacao
        velocidadeAtual += (velocidadeAlvo - velocidadeAtual) * 0.3;
        velocidadeEl.textContent = Math.round(velocidadeAtual);

        animacaoId = requestAnimationFrame(animarVelocidade);
    }

    // ============================================================
    // Funcao para definir a velocidade com transicao suave
    // ============================================================
    function definirVelocidade(valor) {
        velocidadeAlvo = valor;
        if (!animacaoId) {
            animacaoId = requestAnimationFrame(animarVelocidade);
        }
    }

    // ============================================================
    // Funcao para atualizar a marcha exibida
    // ============================================================
    function definirMarcha(valor) {
        if (valor === 0) {
            marchaEl.textContent = 'R';
        } else {
            marchaEl.textContent = valor;
        }
    }

    // ============================================================
    // Funcao para atualizar a barra de RPM
    // ============================================================
    function definirRpm(valor) {
        // valor entre 0.0 e 1.0
        const porcentagem = Math.min(Math.max(valor * 100, 0), 100);
        rpmBarra.style.width = porcentagem + '%';

        // Adiciona efeito pulsante quando RPM esta alto
        if (porcentagem > 85) {
            rpmBarra.classList.add('rpm-alto');
        } else {
            rpmBarra.classList.remove('rpm-alto');
        }
    }

    // ============================================================
    // Funcao para atualizar a barra de combustivel
    // ============================================================
    function definirCombustivel(valor) {
        // valor entre 0 e 100
        const porcentagem = Math.min(Math.max(valor, 0), 100);
        combustivelBarra.style.width = porcentagem + '%';
        combustivelValor.textContent = Math.round(porcentagem) + '%';

        // Remove classes anteriores
        combustivelBarra.classList.remove('combustivel-medio', 'combustivel-baixo');

        // Aplica classe baseada no nivel
        if (porcentagem <= 15) {
            combustivelBarra.classList.add('combustivel-baixo');
        } else if (porcentagem <= 35) {
            combustivelBarra.classList.add('combustivel-medio');
        }
    }

    // ============================================================
    // Funcao para atualizar indicadores (motor, luzes)
    // ============================================================
    function definirIndicador(elemento, ativo) {
        if (ativo) {
            elemento.classList.add('ativo');
        } else {
            elemento.classList.remove('ativo');
        }
    }

    // ============================================================
    // Funcao para mostrar o velocimetro
    // ============================================================
    function mostrar() {
        velocimetro.classList.remove('oculto');
    }

    // ============================================================
    // Funcao para esconder o velocimetro
    // ============================================================
    function esconder() {
        velocimetro.classList.add('oculto');

        // Reseta os valores ao esconder
        setTimeout(function () {
            velocidadeAtual = 0;
            velocidadeAlvo = 0;
            velocidadeEl.textContent = '0';
            definirMarcha(1);
            definirRpm(0);
            definirCombustivel(100);
            definirIndicador(indMotor, false);
            definirIndicador(indLuzes, false);
        }, 400);
    }

    // ============================================================
    // Listener para mensagens do NUI (enviadas pelo client.lua)
    // ============================================================
    window.addEventListener('message', function (event) {
        var dados = event.data;

        switch (dados.action) {
            case 'mostrar':
                mostrar();
                break;

            case 'esconder':
                esconder();
                break;

            case 'atualizar':
                definirVelocidade(dados.velocidade);
                definirRpm(dados.rpm);
                definirMarcha(dados.marcha);
                definirCombustivel(dados.combustivel);
                definirIndicador(indMotor, dados.motor);
                definirIndicador(indLuzes, dados.luzes);
                break;
        }
    });
})();
